<?php

class zapilyator extends base_module {
    private $space = array(
        0 => 40873,
        1 => 16384,
        3 => 16384,
        4 => 16384,
        6 => 16384,
        7 => 16384
    );

    private $slideshowPages = array(); // Страницы, содержащие слайды (не должны упаковываться через hrust13)

    private $sizes = array(
        'Int flow' => 346,
        // Фиксированный код поддержки MusicBank в page 0 (opros/initmus и обвязка)
        'MusicBank code' => 322,

        /* Player length:
         * 2840 - player
         * 0003 - init
         * 0017 - play
         * 0002 - CUR_PATTERN / _curPattern
         * 0255 - safety align #100
         */
        'PT3 player' => 3117,
    );

    var $isOverflow = false; // Size overflowing control

    private $cacheDir = 'var/cache/';

    function __construct($config) {
        parent::__construct();

        if (isset($config['cacheDir'])) {
            $this->cacheDir = $config['cacheDir'];
        } else {
            $this->cacheDir = PROJECT_ROOT . $this->cacheDir;

            if (!file_exists($this->cacheDir)) {
                mkdir($this->cacheDir, 0777, true);
            }
        }
    }

    private function emptyFolder($dir, $liveTime = 0) {
        if (substr($dir, -1, 1) != '/') {
            $dir = $dir . '/';
        }

        // Remove old files
        $files = scandir($dir);
        foreach ($files as $f) {
            if (is_dir($dir . $f)) {
                continue;
            }

            if ((time() - filemtime($dir . $f) > $liveTime) && is_writable($dir . $f)) {
                @unlink($dir . $f);
            }
        }
    }

    private function allocSpace($size, $page = null) {
        // Alloc in given page
        if ($page !== null && isset($this->space[$page])) {
            if ($this->space[$page] < $size) {
                $this->isOverflow = true;
                return false;
            }

            $this->space[$page] -= $size;
            return $page;
        }

        // Alloc in page with best fit (наименьший подходящий блок)
        $found_size = 65535;
        $found_page = false;
        foreach ($this->space as $page => $free) {
            if ($free >= $size && $free < $found_size) {
                $found_size = $free;
                $found_page = $page;
            }
        }

        if ($found_page === false) {
            $this->isOverflow = true;
            return false;
        }

        $this->space[$found_page] -= $size;
        return $found_page;
    }

    private function zx_screen_address($x, $y) {
        $x0 = $x - 1;
        $y0 = $y - 1;

        return 0x4000
             + (($y0 & 0x07) << 5)       // строка внутри блока (0..7) × 32
             + (($y0 >> 3) * 0x0800)     // номер блока (0,1,2) × 2048
             + $x0;
    }

    private function getSnippet($snippet_name, $options = array()) {
        if (!$xml = simplexml_load_file(PROJECT_ROOT . 'resources/output/snippets/' . $snippet_name . '.xml')) {
            $this->error('Wrong XML file.', __FILE__, __LINE__);
            return false;
        }

        $snippet = array(
            'template' => (string)$xml->template,
            'length' => (int)$xml->length
        );

        foreach ($xml->params->param as $param) {
            $varName = (string)$param->varname;
            $value = isset($options['params'][$varName]) ? $options['params'][$varName] : (string)$param->default;
            $snippet['template'] = str_replace('%' . $varName . '%', $value, $snippet['template']);
        }

        // Set module name
        if (isset($options['module']) && $options['module']) {
            $snippet['template'] = "\t" . 'module ' . $options['module'] . "\n" . $snippet['template'] . "\n\t" . 'endmodule';
        }

        // Switch page
        if (isset($options['page'])) {
            $snippet['template'] = "\t" . 'ld a, #' . sprintf("%02x", $options['page'] + 0x10) . ' : call setPage' . "\n" . $snippet['template'];
            $snippet['length'] += 5;
        }

        // Set function name
        if (isset($options['function_name'])) {
            $snippet['template'] = $options['function_name'] . "\n" . $snippet['template'];
        }

        return $snippet;
    }

    private function generateAnalyzerData($scr_filename) {
        $scr = file_get_contents($scr_filename);
        $result = array();

        for ($address = 6144; $address < 6912; $address++) {
            if (!isset($scr[$address]) || ord($scr[$address]) < 128) continue;

            $scr[$address] = chr(ord($scr[$address]) - 128);
            $result[] = "\t" . 'dw #' . sprintf("%04x", $address + 0x4000);
        }

        if (empty($result)) return false;

        // Save modified (without FLASH) screen
        file_put_contents($scr_filename, $scr);

        return $result;
    }

    private function generateTimeline($timeline) {
        $timeline_flow = $functions = '';

        foreach ($timeline as $key => $t) {
            $function_name = isset($t['function_name']) ? $t['function_name'] : 'INTFLOW' . $key;
            $next_run = $t['next_run'] === 'next' ? $key + 1 : $t['next_run'];

            $timeline_flow .= "\t" . 'db #' . sprintf("%02x", $t['star_pattern']) . '		; start pattern' . "\n";
            $timeline_flow .= "\t" . 'db #' . sprintf("%02x", $t['page'] + 0x10) . '		; proc page' . "\n";
            $timeline_flow .= "\t" . 'dw ' . $function_name . '		; proc address' . "\n";
            $timeline_flow .= "\t" . 'dw #' . sprintf("%04x", $t['ints_counter']) . '		; ints counter' . "\n";
            $timeline_flow .= "\t" . 'db #' . sprintf("%02x", $t['stop_pattern']) . '		; stop pattern' . "\n";
            $timeline_flow .= "\t" . 'db #' . sprintf("%02x", $next_run) . '		; next run' . "\n";
            $timeline_flow .= "\n";

            if (isset($t['function']) && $t['function']) {
                $functions .= $function_name . "\n";
                $functions .= $t['function'];
                $functions .= "\n";
            }
        }

        return array($timeline_flow, $functions);
    }

    private function generateMainFlow($main_flow) {
        $result = array();

        foreach ($main_flow as $i) {
            if (!is_array($i)) {
                $result[] = $i;
                continue;
            }

            // Call from different page
            $result[] = "\t" . 'ld a, #' . sprintf("%02x", $i['page'] + 0x10) . ' : call setPage';
            $result[] = $i['code'];
            $result[] = "\t" . 'ld a, #10 : call setPage';
        }

        return implode("\n", $result);
    }

    private function generateDataFlow($data_flow) {
        
        $result = array();
        foreach ($data_flow as $page => $data) {
            // Фейковая 8-я страница
            if ($page == 8 && !empty($data)) {
                $result[] = '/* overflow data';
                $result = array_merge($result, $data);
                $result[] = '*/';
                continue;
            }

            // Заголовок для всех непустых страниц, кроме 0-й
            if ($page != 0 && !empty($data)) {
                $result[] = "\t" . 'define _page' . $page . ' : page ' . $page . ' : org #c000';
                $result[] = 'page' . $page . 's';
            }

            // Завершение для всех непустых страниц, для 0-й - всегда
            if ($page == 0 || !empty($data)) {
                $result = array_merge($result, $data);
                $result[] = 'page' . $page . 'e' . "\t" . 'display /d, \'Page ' . $page . ' free: \', #ffff - $';
            }
        }

        return implode("\n", $result);
    }

    function loadProject($projectName) {
        if (!file_exists($this->cacheDir . $projectName)) {
            $this->error('System error: file "'.$projectName.'" not found in '.$this->cacheDir, __FILE__, __LINE__);
            return false;
        }

        if (!$data = NFW::i()->unserializeArray(file_get_contents($this->cacheDir . $projectName))) {
            $this->error('System error: unable to reload project', __FILE__, __LINE__);
            return false;
        }

        // Инициализация int_counter_enabled по умолчанию для старых проектов
        if (!isset($data['int_counter_enabled'])) {
            $data['int_counter_enabled'] = '0';
        }

        return $data;
    }

    function saveProject($projectName, $data) {
        if (file_put_contents($this->cacheDir . $projectName, NFW::i()->serializeArray($data)) === false) {
            $this->error('System error: save file "'.$projectName.'" into '.$this->cacheDir.' failed', __FILE__, __LINE__);
            return false;
        }

        return true;
    }

    function parseAnimation($data, $i, $method = ZXAnimation::METHOD_FAST) {
        // Устанавливаем таймаут для предотвращения зависания
        set_time_limit(30); // 30 секунд максимум
        
        try {
        // Parse GIF portion
        $parser = new parse256x192(array('initialColor' => $data['main']['color'], 'sourceType' => $data[$i]['source_type'], 'defaultDuration' => $data[$i]['speed']));
            
            // Проверяем существование файла
            if (!file_exists($data[$i]['source'])) {
    
                NFW::i()->renderJSON(array('result' => 'error', 'last_msg' => 'Animation source file not found'));
                return false;
            }
            
        if (!$loading_result = $parser->load($data[$i]['source'], array(
            'from' => isset($data['from']) ? $data['from'] : 0,
            'count' => 100,
            'is_continuous' => true
        ))) {
    
            NFW::i()->renderJSON(array('result' => 'error', 'last_msg' => $parser->last_msg));
                return false;
        }

        $frames = $parser->parseSource();
            
            if (empty($frames)) {
    
                NFW::i()->renderJSON(array('result' => 'error', 'last_msg' => 'No frames parsed from animation'));
                return false;
            }

        // Generate data
        $generator = new ZXAnimation();
        $result = $generator->generateCode($frames, $method);
        $data[$i]['parsed'] = array_merge($data[$i]['parsed'], $result);
        $data[$i]['totalFramesLen'] += $generator->totalFramesLen;
        $data[$i]['totalBytesAff'] += $generator->totalBytesAff;

        return array($data, $loading_result);
            
        } catch (Exception $e) {

            NFW::i()->renderJSON(array('result' => 'error', 'last_msg' => 'Animation parsing error: ' . $e->getMessage()));
            return false;
        }
    }

    function getFreeSpace() {
        $free_space = 0;
        foreach ($this->space as $val) {
            $free_space += $val;
        }

        return $free_space;
    }

    function getFreeSpaceByPage() {
        $result = array();
        foreach ($this->space as $page => $free) {
            $result[$page] = $free;
        }
        return $result;
    }

    function upload($field_name) {
        $this->error_report_type = 'active_form';
        $this->error = false;
        $this->emptyFolder($this->cacheDir, 3600);

        if (!isset($_FILES[$field_name])) {
            $this->error('No file selected: ' . $field_name, __FILE__, __LINE__);
            return false;
        }

        $file = $_FILES[$field_name];

        if (!empty($file['error'])) {
            switch ($file['error']) {
                case '1':
                    $this->error('The uploaded file exceeds the upload_max_filesize directive in php.ini');
                    return false;
                case '2':
                    $this->error('The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form');
                    return false;
                case '3':
                    $this->error('The uploaded file was only partially uploaded');
                    return false;
                case '4':
                    $this->error('No file was uploaded.');
                    return false;
                case '6':
                    $this->error('Missing a temporary folder');
                    return false;
                case '7':
                    $this->error('Failed to write file to disk');
                    return false;
                case '8':
                    $this->error('File upload stopped by extension');
                    return false;
                case '999':
                default:
                    $this->error('No error code available');
                    return false;
            }
        }

        $targetFile = $this->cacheDir . md5($file['name'] . $file['size']);

        if (file_exists($targetFile) && !is_writable($targetFile)) {
            $this->error('File exists and can not be overwritten');
            return false;
        }

        move_uploaded_file(urldecode($file['tmp_name']), $targetFile);
        chmod($targetFile, 0777);

        return array(
            'targetFile' => $targetFile,
            'fileExtension' => pathinfo($file['name'], PATHINFO_EXTENSION)
        );
    }

    function generateSource($params, $projectName = null) {
        // Generate ZIP
        $dest_filename = md5(serialize($params));
        $resSizes = [];
        $zip = new ZipArchive();
        $zip->open($this->cacheDir . $dest_filename, ZIPARCHIVE::OVERWRITE | ZIPARCHIVE::CREATE);
        $zip->addFile(PROJECT_ROOT . 'resources/output/make.cmd', 'make.cmd');
        
        $zip->addFile(PROJECT_ROOT . 'resources/output/sources/builder.asm', 'sources/builder.asm');
        // laser521.asm будет добавлен условно, если используется background

        // -----------------
        //  Generate source
        // -----------------

        $fp = fopen(PROJECT_ROOT . 'resources/output/sources/zapil.asm.tpl', 'r');
        $source_tpl = fread($fp, filesize(PROJECT_ROOT . 'resources/output/sources/zapil.asm.tpl'));
        fclose($fp);

        $data_flow = $main_flow = $timeline = array();
        // Сохраняем начальные значения свободного места по страницам,
        // чтобы позже корректно вычислять смещение для музыки (после анимации)
        $initialSpace = $this->space;
        $is_music = isset($params['music_file']);
        $is_musicbank = isset($params['musicbank']) && isset($params['musicbank']['asm_table']) && isset($params['musicbank']['music_files']);
        $has_music = $is_music || $is_musicbank; // Универсальная проверка наличия музыки
        $use_laser521 = false; // Флаг для использования laser521
        
        // Если есть MusicBank, то нужен laser521 для распаковки
        if ($is_musicbank) {
            $use_laser521 = true;
        }
        
        // Инициализация int_counter_enabled по умолчанию
        if (!isset($params['int_counter_enabled'])) {
            $params['int_counter_enabled'] = '0';
        }

        // Generate empty $data_flow array
        foreach ($this->space as $page => $foo) $data_flow[$page] = array();

        // -- music
        if ($is_music) {
            $source_tpl = str_replace('%if_music%', '', $source_tpl);

            $this->allocSpace($this->sizes['PT3 player'] + filesize($params['music_file']), 0);
            $zip->addFile(PROJECT_ROOT . 'resources/output/sources/PTxPlay.asm', 'sources/PTxPlay.asm');
            $zip->addFile($params['music_file'], 'res/music');
        } else
            $source_tpl = str_replace('%if_music%', ';', $source_tpl);

        // -- musicbank
        if ($is_musicbank) {
            $source_tpl = str_replace('%if_musicbank%', '', $source_tpl);
            $this->allocSpace($this->sizes['PT3 player'], 0);
            // Учёт базового кода MusicBank (инициализация/опроc)
            $this->allocSpace($this->sizes['MusicBank code'], 0);
            // Добавляем PTxPlay.asm только если он еще не добавлен
            if (!$is_music) {
                $zip->addFile(PROJECT_ROOT . 'resources/output/sources/PTxPlay.asm', 'sources/PTxPlay.asm');
            }
        } else {
            $source_tpl = str_replace('%if_musicbank%', ';', $source_tpl);
        }

        // Инициализация имени анимации в main_flow (по умолчанию ANIMATION1)
        $source_tpl = str_replace('%ANIMATION_MAIN_FLOW_NAME%', 'ANIMATION1', $source_tpl);
        
        // Инициализация флага musicbank (по умолчанию отключен)
        $source_tpl = str_replace('%if_play_musicbank%define _PLAY_MUSICBANK', ';define _PLAY_MUSICBANK', $source_tpl);

        // Set border
        $snippet = $this->getSnippet('set_border', array(
            'params' => array('VALUE' => $params['splash']['border'])
        ));
        $main_flow[] = $snippet['template'];
        $this->allocSpace($snippet['length'], 0);

        // -- splash background
        if (isset($params['splash']['background'])) {
            // Define _SPLASH_BG for builder.asm
            $source_tpl = str_replace('%if_splash_bg%', 'define _SPLASH_BG', $source_tpl);
            // Splash background НЕ использует laser521.asm (загружается через call #3d13)
            
            // === LASER521 PACKING ===
            $splash_src = $params['splash']['background'];
            $splash_pck = $splash_src . '.pck';
            $laser_path = PROJECT_ROOT . 'resources/output/laser521.exe';
            exec(escapeshellarg($laser_path) . ' -d ' . escapeshellarg($splash_src) . ' ' . escapeshellarg($splash_pck));

            $zip->addFile($splash_pck, 'res/splash_bg.pck');
            $resSizes['splash_bg.pck'] = filesize($splash_pck);

            // Generate pause
            if ($has_music) {
                $snippet = $this->getSnippet('wait_pattern', array(
                    'module' => 'splash_delay',
                    'params' => array('DELAY' => $params['splash']['delay'])
                ));
            } else {
                $snippet = $this->getSnippet('pause_short', array(
                    'params' => array('DELAY' => '#FF')
                ));
            }
            $main_flow[] = $snippet['template'];
            $this->allocSpace($snippet['length'], 0);
        } else {
            // No splash background - comment out the define
            $source_tpl = str_replace('%if_splash_bg%', '; define _SPLASH_BG', $source_tpl);
        }



        // Change border if different
        if ($params['main']['border'] != $params['splash']['border']) {
            $snippet = $this->getSnippet('set_border', array(
                'params' => array('VALUE' => $params['main']['border'])
            ));
            $main_flow[] = $snippet['template'];
            $this->allocSpace($snippet['length']);
        }

        // Change background after splash
        if (isset($params['main']['background'])) {
            $use_laser521 = true; // Main background использует laser521.asm для распаковки
            // === LASER521 PACKING ===
            $main_src = $params['main']['background'];
            $main_pck = $main_src . '.pck';
            $laser_path = PROJECT_ROOT . 'resources/output/laser521.exe';
            exec(escapeshellarg($laser_path) . ' ' . escapeshellarg($main_src) . ' ' . escapeshellarg($main_pck));

            $snippet = $this->getSnippet('copy_to_scr_zx7', array(
                'params' => array('SOURCE' => 'MAIN_BG')
            ));
            
            // Если есть musicbank, проверяем, не занята ли page1 музыкой
            $main_bg_size = filesize($main_pck) + $snippet['length'];
            if (isset($params['musicbank']) && isset($params['musicbank']['music_files'])) {
                $page1_used = false;
                foreach ($params['musicbank']['music_files'] as $music_file) {
                    if ($music_file['bank'] == 17) { // Банк 17 = page1
                        $page1_used = true;
                        break;
                    }
                }
                if ($page1_used) {
                    // Если page1 занята музыкой, ищем другую страницу
                    $page = false;
                    foreach ($this->space as $p => $free) {
                        if ($p != 1 && $free >= $main_bg_size) {
                            $page = $p;
                            $this->space[$p] -= $main_bg_size;
                            break;
                        }
                    }
                    if ($page === false) {
                        $page = $this->allocSpace($main_bg_size); // Fallback
                    }
                } else {
                    $page = $this->allocSpace($main_bg_size);
                }
            } else {
                $page = $this->allocSpace($main_bg_size);
            }
            
            $data_flow[$page][] = 'MAIN_BG' . "\t" . 'incbin "res/main_bg.pck"';

            $main_flow[] = "\t" . 'ld a, ' . ($page + 0x10) . ' : call setPage' . "\n" . $snippet['template'];
            $this->allocSpace(5, 0);

            $zip->addFile($main_pck, 'res/main_bg.pck');
            $resSizes['main_bg.pck'] = filesize($main_pck);
        } else {
            // Simple clear screen
            $snippet = $this->getSnippet('clear_scr', array(
                'params' => array('ATTR' => $params['main']['color'])
            ));
            $main_flow[] = $snippet['template'];
            $this->allocSpace($snippet['length'], 0);
        }

        // Analyzer in main screen
        if (isset($params['main']['analyzer']['enabled']) && $params['main']['analyzer']['enabled']) {
            $analyzer_type = isset($params['main']['analyzer']['type']) ? $params['main']['analyzer']['type'] : 'goba_left';
            $snippet_name = 'analyzer_' . $analyzer_type;
            
            $snippet_params = array();
            
            // Добавляем специфичные параметры для goba анализаторов
            if ($analyzer_type == 'goba_left') {
                // Используем analyzer_goba_left с параметрами x, y
                $x = isset($params['main']['analyzer']['x']) ? $params['main']['analyzer']['x'] : 1;
                $y = isset($params['main']['analyzer']['y']) ? $params['main']['analyzer']['y'] : 1;
                
                // Ограничиваем значения
                $x = max(1, min(32, $x));
                $y = max(1, min(24, $y));
                
                $snippet_name = 'analyzer_goba_left';
                $snippet_params['SCR_AY1'] = sprintf("#%04x", $this->zx_screen_address($x, $y));
                $snippet_params['SCR_AY2'] = sprintf("#%04x", $this->zx_screen_address($x, $y + 1));
                $snippet_params['SCR_AY3'] = sprintf("#%04x", $this->zx_screen_address($x, $y + 2));
            } elseif ($analyzer_type == 'goba_right') {
                // Используем analyzer_goba_right с параметрами x, y
                $x = isset($params['main']['analyzer']['x']) ? $params['main']['analyzer']['x'] : 1;
                $y = isset($params['main']['analyzer']['y']) ? $params['main']['analyzer']['y'] : 1;
                
                // Ограничиваем значения
                $x = max(1, min(32, $x));
                $y = max(1, min(24, $y));
                
                $snippet_name = 'analyzer_goba_right';
                $snippet_params['SCR_AY1'] = sprintf("#%04x", $this->zx_screen_address($x, $y));
                $snippet_params['SCR_AY2'] = sprintf("#%04x", $this->zx_screen_address($x, $y + 1));
                $snippet_params['SCR_AY3'] = sprintf("#%04x", $this->zx_screen_address($x, $y + 2));
            } elseif ($analyzer_type == 'goba_up') {
                // Используем analyzer_goba_up с параметрами x, y
                $x = isset($params['main']['analyzer']['x']) ? $params['main']['analyzer']['x'] : 1;
                $y = isset($params['main']['analyzer']['y']) ? $params['main']['analyzer']['y'] : 1;
                
                // Ограничиваем значения
                $x = max(1, min(32, $x));
                $y = max(1, min(24, $y));
                
                $snippet_name = 'analyzer_goba_up';
                $snippet_params['SCR_AY1'] = sprintf("#%04x", $this->zx_screen_address($x, $y));
                $snippet_params['SCR_AY2'] = sprintf("#%04x", $this->zx_screen_address($x, $y + 1));
                $snippet_params['SCR_AY3'] = sprintf("#%04x", $this->zx_screen_address($x, $y + 2));
            } elseif ($analyzer_type == 'goba_down') {
                // Используем analyzer_goba_down с параметрами x, y
                $x = isset($params['main']['analyzer']['x']) ? $params['main']['analyzer']['x'] : 1;
                $y = isset($params['main']['analyzer']['y']) ? $params['main']['analyzer']['y'] : 1;
                
                // Ограничиваем значения
                $x = max(1, min(32, $x));
                $y = max(1, min(24, $y));
                
                $snippet_name = 'analyzer_goba_down';
                $snippet_params['SCR_AY1'] = sprintf("#%04x", $this->zx_screen_address($x, $y));
                $snippet_params['SCR_AY2'] = sprintf("#%04x", $this->zx_screen_address($x, $y + 1));
                $snippet_params['SCR_AY3'] = sprintf("#%04x", $this->zx_screen_address($x, $y + 2));
            } elseif ($analyzer_type == 'bright') {
                $snippet_name = 'analyzer_bright';
            } else {
                $snippet_name = 'analyzer_goba_left';
            }
            
            $snippet = $this->getSnippet($snippet_name, array(
                'module' => 'main_analyzer',
                'params' => $snippet_params
            ));

            // Generate main effect directly (without pause)
            $timeline[] = array(
                'star_pattern' => $params['splash']['delay'],
                'page' => 0,
                'function' => $snippet['template'],
                'ints_counter' => 0xffff, // 0xffff для бесконечного эффекта (будет сброшен assembly кодом)
                'stop_pattern' => 0xff, // бесконечно
                'next_run' => 0xff
            );
            

            $data_size = 0; // Goba анализаторы не используют дополнительные данные
            $this->allocSpace($snippet['length'] + $data_size, 0);
        }

        // Предварительный анализ размеров анимаций для оптимизации
        $animation_sizes = [];
        for ($i = 1; $i <= 4; $i++) {
            if (!$params[$i]) continue;
            $total_size = 0;
            foreach ($params[$i]['parsed'] as $frame) {
                $total_size += $frame['frame_len'];
            }
            $tmp_snippet = $this->getSnippet('animation_' . $params[$i]['method']);
            $animation_sizes[$i] = $total_size + $tmp_snippet['length'] + count($params[$i]['parsed']) * 4;
        }
        
        // Сортируем анимации по размеру (большие сначала для лучшего размещения)
        // НЕ резервируем место под музыку заранее - сначала размещаем анимацию

        arsort($animation_sizes);





        // Generate animations after music
        foreach ($animation_sizes as $i => $size) {
            if (!$params[$i]) continue;

            // Alloc memory for animation
            $tmp_snippet = $this->getSnippet('animation_' . $params[$i]['method']);
            $this->allocSpace($tmp_snippet['length'] + count($params[$i]['parsed']) * 4, 0);

            // Generate page-related array of frames,
            // create `diff` directory
            // and create DB array
            $anima_frames = array();
            foreach ($params[$i]['parsed'] as $key => $frame) {
                // Размещаем анимацию в любой доступной странице
                    $page = $this->allocSpace($frame['frame_len']);
                    if ($page === false) {
                        $page = 8;    // Fake page for overflowed frames
                }
                


                $proc_name = 'A' . $i . '_' . $page . '_' . sprintf("%04x", $key);


                $data_flow[$page][] = $proc_name . "\t" . 'include "res/' . $proc_name . '.asm"';
                $zip->addFromString('res/' . $proc_name . '.asm', $frame['source']);

                $anima_frames[] = ($page == 8 ? ';' : '') . "\tdb " . $frame['duration'] . ', ' . $page . ' : dw ' . $proc_name;

                // Adding `diff`
                if (empty($frame['diff'])) continue;
                $diff = '';
                foreach ($frame['diff'] as $address => $byte) {
                    $diff .= sprintf("%04x", $address) . ' ' . sprintf("%02x", $byte) . "\n";
                }
                $zip->addFromString('diff/' . $i . '-' . sprintf("%04d", $key) . '.txt', $diff);
            }

            // Generate animation function
            $snippet = $this->getSnippet('animation_' . $params[$i]['method'], array(
                'module' => 'animation' . $i,
                'function_name' => 'ANIMATION' . $i,
                'params' => array(
                    'ANIMATION_FRAMES' => implode("\n", $anima_frames)
                )
            ));

            if ($params[$i]['position'] == 'main_flow') {
                // Generate main flow function (at first of flow!)
                array_unshift($data_flow[0], $snippet['template']);
                // Устанавливаем флаг для анимации в main_flow
                $source_tpl = str_replace('%if_animation_main_flow%define _ANIMATION_MAIN_FLOW', 'define _ANIMATION_MAIN_FLOW', $source_tpl);
                // Заменяем имя анимации в main_flow
                $source_tpl = str_replace('%ANIMATION_MAIN_FLOW_NAME%', 'ANIMATION' . $i, $source_tpl);
                $this->allocSpace(6, 0);
            } else {
                // Generate pause before start
                $timeline[] = array(
                    'star_pattern' => $params['splash']['delay'],
                    'page' => 0,
                    'function' => "\t" . 'ret' . "\n",
                    'ints_counter' => 0x0004,
                    'stop_pattern' => 0xff, // бесконечно
                    'next_run' => 'next'
                );
                $this->allocSpace(1, 0);

                // Generate timeline function
                $timeline[] = array(
                    'star_pattern' => 0xfe,
                    'page' => 0,
                    'function' => $snippet['template'],
                    'ints_counter' => 0xffff, // 0xffff для бесконечного эффекта (будет сброшен assembly кодом)
                    'stop_pattern' => 0xff, // бесконечно
                    'next_run' => 0xff
                );
            }
        }



        // Инициализация флага анимации в main_flow (по умолчанию отключен)
        $source_tpl = str_replace('%if_animation_main_flow%define _ANIMATION_MAIN_FLOW', ';define _ANIMATION_MAIN_FLOW', $source_tpl);

        // -- scroll (after animations)
        if (isset($params['scroll']['text'])) {
            // Получаем ширину скролла
            $scroll_width = isset($params['scroll']['width']) ? $params['scroll']['width'] : 32;
            
            // Получаем смещение скролла и рассчитываем его от width и position
            $scroll_offset = isset($params['scroll']['offset']) ? $params['scroll']['offset'] : 0;
            
            // Рассчитываем адрес заливки с учетом смещения (offset применяется к базовому адресу)
            $base_attr_address = hexdec(substr($params['scroll']['attr'], 1)); // Убираем # и конвертируем
            $attr_address_with_offset = '#' . sprintf('%04x', $base_attr_address + $scroll_offset);
            

            
            // Prepare screen background for scroll with offset
            // Для скролла 8px применяем окраску цветом только если НЕТ фонового изображения, для 16px - не применяем
            if (isset($params['scroll']['type']) && $params['scroll']['type'] == 8 && !isset($params['main']['background'])) {
                $snippet = $this->getSnippet('fill_scroll_attr', array(
                'params' => array(
                    'FROM' => $params['scroll']['attr'],
                    'WIDTH' => $scroll_width,
                    'FILL' => $params['scroll']['color'],
                        'OFFSET' => $scroll_offset
                )
            ));
            $main_flow[] = $snippet['template'];
            $this->allocSpace($snippet['length'], 0);
            }

            // Generate scroll function
            if (isset($params['scroll']['type']) && $params['scroll']['type'] == 8) {
                // Для скролла 8 выбираем направление
                $scroll_direction = isset($params['scroll']['direction']) ? $params['scroll']['direction'] : 'left';
                $scroll_type = ($scroll_direction == 'up') ? 'scroll_text8_offset_up' : 'scroll_text8_offset';
                
            } else {
                $scroll_type = 'scroll_text16_offset';
            }
            $font_name = isset($params['scroll']['type']) && $params['scroll']['type'] == 8 ? 'FONT8X8' : 'FONT16X16';
            
            // Используем адрес скролла без применения offset (offset применяется только к атрибутам)
            $calculated_address = $params['scroll']['address'];
            

            
            // Подготавливаем параметры в зависимости от типа сниппета
            if ($scroll_type == 'scroll_text8_offset_up') {
                // Для скролла вверх нужны координаты X, Y и высота
                
                $snippet_params = array(
                    'COORD_X' => isset($params['scroll']['coord_x']) ? $params['scroll']['coord_x'] : 16,
                    'COORD_Y' => isset($params['scroll']['coord_y']) ? $params['scroll']['coord_y'] : 4,
                    'VISOTA' => isset($params['scroll']['visota']) ? $params['scroll']['visota'] : 8
                );
                
            } else {
                // Для обычного скролла используем старые параметры
                $snippet_params = array(
                    'ADDRESS' => $calculated_address,
                    'WIDTH' => $scroll_width,
                    'OFFSET' => $scroll_offset
                );
            }
            
            $snippet = $this->getSnippet($scroll_type, array(
                'function_name' => 'SCROLL_FUNC',
                'params' => $snippet_params
            ));

            // Разный размер буфера для разных типов скролла
            $buffer_size = (isset($params['scroll']['type']) && $params['scroll']['type'] == 8) ? 8 : 32;
            // Определяем реальный размер шрифта
            if (isset($params['scroll']['type']) && $params['scroll']['type'] == 8) {
                if (isset($params['scroll']['font']) && $params['scroll']['font'] == 'custom_8x8font' && isset($params['scroll']['custom_font_file'])) {
                    $font_size = @filesize($params['scroll']['custom_font_file']);
                } else {
                    $font_size = @filesize(PROJECT_ROOT . 'resources/output/res/8x8font');
                }
            } else {
                if (isset($params['scroll']['font']) && $params['scroll']['font'] == 'custom_16x16font' && isset($params['scroll']['custom_font_file'])) {
                    $font_size = @filesize($params['scroll']['custom_font_file']);
                } else {
                    $font_file_sel = isset($params['scroll']['font']) ? $params['scroll']['font'] : '16x16font1';
                    $font_path = PROJECT_ROOT . 'resources/output/res/' . $font_file_sel;
                    $font_data_tmp = @file_get_contents($font_path);
                    $font_size = $font_data_tmp === false ? 0 : strlen($font_data_tmp);
                }
            }
            if ($font_size === false || $font_size < 0) { $font_size = 0; }
            $total_size = $snippet['length'] + $buffer_size + strlen($params['scroll']['text']) + 1 + (int)$font_size;
            
            // Если есть musicbank, размещаем скролл в Page 0 (как для 8px, так и для 16px)
            if (isset($params['musicbank'])) {
                $page = $this->allocSpace($total_size, 0);
                if ($page === false) {
                    // Если Page 0 переполнена, ищем любую доступную страницу
                    $page = $this->allocSpace($total_size);
                }
            } else {
                // Для скролла 8px размещаем в Page 0, для 16px - в любой доступной странице
            if (isset($params['scroll']['type']) && $params['scroll']['type'] == 8) {
                    $page = $this->allocSpace($total_size, 0);
                    if ($page === false) {
                        // Если Page 0 переполнена, ищем любую доступную страницу
                        $page = $this->allocSpace($total_size);
                    }
                } else {
                    $page = $this->allocSpace($total_size);
                }
            }
            $data_flow[$page][] = $snippet['template'];
            $data_flow[$page][] = 'SCROLL_BUFF' . "\t" . 'block ' . $buffer_size;
            $data_flow[$page][] = 'SCROLL_TEXT' . "\t" . 'incbin "res/scroll"';
            $data_flow[$page][] = "\t" . 'db #00';
            // Выбираем правильный файл шрифта в зависимости от типа
            $font_file = (isset($params['scroll']['type']) && $params['scroll']['type'] == 8) ? '8x8font' : '16x16font';
            $data_flow[$page][] = $font_name . "\t" . 'incbin "res/' . $font_file . '"';

            // Generate scroll entry directly (without pause)
            $scroll_entry = array(
                'star_pattern' => $params['splash']['delay'],
                'page' => $page,
                'function_name' => 'SCROLL_FUNC',
                'ints_counter' => 0xffff, // 0xffff для бесконечного эффекта (будет сброшен assembly кодом)
                'stop_pattern' => 0xff, // скролл всегда бесконечный
                'next_run' => 0xff
            );
            
            // init_up вызывается в основном коде при инициализации, не нужно добавлять в timeline

            // Вставляем скролл перед анимациями
            array_unshift($timeline, $scroll_entry);

            $scrollData = iconv("UTF-8", 'cp1251', mb_strtoupper($params['scroll']['text'], "UTF-8"));
            $zip->addFromString('res/scroll', $scrollData);
            $resSizes['scroll'] = strlen($scrollData);
            
            // Добавляем файл шрифта для скролла
            if (isset($params['scroll']['type']) && $params['scroll']['type'] == 8) {
                if (isset($params['scroll']['font']) && $params['scroll']['font'] == 'custom_8x8font') {
                    $zip->addFile($params['scroll']['custom_font_file'], 'res/8x8font');
                    $resSizes['8x8font'] = filesize($params['scroll']['custom_font_file']);
                } else {
                    $zip->addFile(PROJECT_ROOT . 'resources/output/res/8x8font', 'res/8x8font');
                    $resSizes['8x8font'] = filesize(PROJECT_ROOT . 'resources/output/res/8x8font');
                }
            } else {
                if (isset($params['scroll']['font']) && $params['scroll']['font'] == 'custom_16x16font') {
                    $zip->addFile($params['scroll']['custom_font_file'], 'res/16x16font');
                    $resSizes['16x16font'] = filesize($params['scroll']['custom_font_file']);
                } else {
                    $font_file = isset($params['scroll']['font']) ? $params['scroll']['font'] : '16x16font1';
                    $font_data = file_get_contents(PROJECT_ROOT . 'resources/output/res/' . $font_file);
                    $zip->addFromString('res/16x16font', $font_data);
                    $resSizes['16x16font'] = strlen($font_data);
                }
            }
        }

        // Устанавливаем флаг для скролла up
        if (isset($params['scroll']['direction']) && $params['scroll']['direction'] == 'up') {
            $source_tpl = str_replace('%if_scroll_up%', '', $source_tpl);
        } else {
            $source_tpl = str_replace('%if_scroll_up%', ';', $source_tpl);
        }

        // Устанавливаем флаг для счетчика прерываний
        if (isset($params['int_counter_enabled']) && $params['int_counter_enabled'] == '1') {
            $source_tpl = str_replace('%if_int_counter%', '', $source_tpl);
        } else {
            $source_tpl = str_replace('%if_int_counter%', ';', $source_tpl);
        }

        // SlideShow: добавляем таблицу и слайды
        if (isset($params['slideshow']) && isset($params['slideshow']['slide_files'])) {

            
            // Упаковываем все слайды через laser521.exe
            $packed_slides = [];
            foreach ($params['slideshow']['slide_files'] as $idx => $slide) {
                $temp_slide_file = tempnam(sys_get_temp_dir(), 'slide');
                file_put_contents($temp_slide_file, $slide['data']);
                
                $packed_slide_file = $temp_slide_file . '.pck';
                $laser_path = PROJECT_ROOT . 'resources/output/laser521.exe';
                exec(escapeshellarg($laser_path) . ' ' . escapeshellarg($temp_slide_file) . ' ' . escapeshellarg($packed_slide_file));
                
                if (file_exists($packed_slide_file)) {
                    $packed_data = file_get_contents($packed_slide_file);
                    $packed_size = strlen($packed_data);
                    
                    $packed_slides[] = [
                        'name' => 'slide' . ($idx + 1) . '.pck',
                        'size' => $packed_size,
                        'data' => $packed_data,
                        'bank' => $slide['bank']
                    ];
                    
                    $zip->addFromString('res/slide' . ($idx + 1) . '.pck', $packed_data);
                    
                    // Очищаем временные файлы
                    unlink($temp_slide_file);
                    unlink($packed_slide_file);
                } else {
                    // Если упаковка не удалась, используем оригинальный файл
                    $packed_slides[] = [
                    'name' => 'slide' . ($idx + 1) . '.bin',
                    'size' => $slide['size'],
                        'data' => $slide['data'],
                        'bank' => $slide['bank']
                ];
                $zip->addFromString('res/slide' . ($idx + 1) . '.bin', $slide['data']);
            }
            }
            
            // Best-Fit упаковка слайдов по страницам памяти
            $allowed_pages = [0, 1, 3, 4, 6, 7];
            $page_to_bank = [0 => 16, 1 => 17, 3 => 19, 4 => 20, 6 => 22, 7 => 23];

            // Для подбора страниц берём фактически оставшееся место по счётчику аллокатора
            $available_by_page = [];
            foreach ($allowed_pages as $page) {
                $available_by_page[$page] = isset($this->space[$page]) ? (int)$this->space[$page] : 0;
            }

            // Сортируем слайды по убыванию размера (большие сначала) для Best-Fit алгоритма
            $sorted_for_placement = $packed_slides;
            usort($sorted_for_placement, function($a, $b) { return $b['size'] - $a['size']; });

            $placed_slides = [];
            foreach ($sorted_for_placement as $slide) {
                $best_page = null;
                $best_leftover = null;
                
                // Сначала пытаемся найти страницу с минимальным остатком (Best-Fit)
                foreach ($allowed_pages as $page) {
                    if ($available_by_page[$page] >= $slide['size']) {
                        $leftover = $available_by_page[$page] - $slide['size'];
                        if ($best_leftover === null || $leftover < $best_leftover) {
                            $best_leftover = $leftover;
                            $best_page = $page;
                        }
                    }
                }
                
                // Если Best-Fit не найден, ищем любую подходящую страницу
                if ($best_page === null) {
                    foreach ($allowed_pages as $page) {
                        if ($available_by_page[$page] >= $slide['size']) {
                            $best_page = $page;
                            break;
                        }
                    }
                }
                
                if ($best_page === null) {
                    continue; // Не помещается никуда
                }

                // Запоминаем назначение страницы
                $placed_slides[] = [
                    'name' => $slide['name'],
                    'size' => $slide['size'],
                    'data' => $slide['data'],
                    'page' => $best_page,
                    'bank' => $page_to_bank[$best_page],
                    'number' => $slide['number']
                ];
                
                // Корректируем реальный счётчик свободного места
                $this->allocSpace($slide['size'], $best_page);
                $available_by_page[$best_page] = isset($this->space[$best_page]) ? (int)$this->space[$best_page] : 0;
            }
            
            // Группируем размещённые слайды по страницам и формируем data_flow
            $slides_by_page = [];
            foreach ($placed_slides as $slide) {
                $slides_by_page[$slide['page']][] = $slide;
            }
            
            foreach ($slides_by_page as $page => $slides) {
                $bank_data = [];
                $cur_addr = 0xC000;
                
                foreach ($slides as $slide) {
                    $bank_data[] = sprintf('    org $%04X', $cur_addr);
                    $bank_data[] = sprintf('    incbin "res/%s"', $slide['name']);
                    $cur_addr += $slide['size'];
                }
                
                        $data_flow[$page] = array_merge($data_flow[$page], $bank_data);
                
                // Помечаем, что эта страница содержит слайды (не должна упаковываться через hrust13)
                $this->slideshowPages[] = $page;
                
                // Создаём маркер в директории проекта для code_compiler.php
                $project_marker = PROJECT_ROOT . "var/cache/zapilyator_page{$page}_slideshow";
                $marker_result = file_put_contents($project_marker, "1");
                
            }
            
            // Учёт места под SlideShowTable в Page 0: по 4 байта на запись + 3 байта терминатор
            $slides_total_count = count($placed_slides);
            if ($slides_total_count > 0) {
                $this->allocSpace($slides_total_count * 4 + 3, 0);
                
                // Сортируем placed_slides по номерам для правильного порядка в таблице
                usort($placed_slides, function($a, $b) { return $a['number'] - $b['number']; });
                
                // Формируем таблицу адресов слайдов с правильными адресами упакованных файлов
                $table_entries = [];
                foreach ($placed_slides as $slide) {
                    // Вычисляем адрес слайда в банке на основе его позиции в data_flow
                    $slide_offset = 0xC000; // Начальный адрес банка
                    
                    // Находим все слайды на той же странице и вычисляем смещение
                    foreach ($placed_slides as $s) {
                        if ($s['page'] === $slide['page'] && $s['name'] === $slide['name']) {
                            break;
                        }
                        if ($s['page'] === $slide['page']) {
                            $slide_offset += $s['size'];
                        }
                    }
                    
                    $table_entries[] = sprintf('    dw $%04X : dw %d', $slide_offset, $slide['bank']);
                }
                $table_entries[] = '    dw $FFFF : db #FF';
                $slideshow_table = "SlideShowTable:\n" . implode("\n", $table_entries);
                
                // Добавляем таблицу в data_flow[0]
                $data_flow[0][] = $slideshow_table;
            }
        }

        // MusicBank: добавляем таблицу и мелодии ПОСЛЕ анимации
        if (isset($params['musicbank']) && isset($params['musicbank']['asm_table']) && isset($params['musicbank']['music_files'])) {
            // Таблица будет добавлена в шаблон после laser521.asm
            
            // Устанавливаем количество музыки для mus_kol
            $music_count = isset($params['musicbank']['music_count']) ? $params['musicbank']['music_count'] : count($params['musicbank']['music_files']);
            $mus_kol_value = sprintf('#%02x', 0x30 + $music_count + 1);
            $source_tpl = str_replace('mus_kol' . "\n" . "\t" . 'cp #3b', 'mus_kol' . "\n" . "\t" . 'cp ' . $mus_kol_value, $source_tpl);
            
            // Включаем флаг _PLAY_MUSICBANK
            $source_tpl = str_replace(';define _PLAY_MUSICBANK', 'define _PLAY_MUSICBANK', $source_tpl);
            

            
            // Best-Fit упаковка музыки по всем оставшимся страницам после анимации
            $music_files = $params['musicbank']['music_files'];
            $items = [];
            foreach ($music_files as $idx => $music) {
                $extension = strtolower(pathinfo($music['name'], PATHINFO_EXTENSION));
                $filename = 'music' . ($idx + 1) . '.' . $extension;
                $items[] = [
                    'order' => $idx,
                    'name' => $filename,
                    'size' => $music['size'],
                    'data' => $music['data']
                ];
                $zip->addFromString('res/' . $filename, $music['data']);
            }
            
            // Разрешённые страницы для музыки и маппинг страниц->банков
            $allowed_pages = [0, 1, 3, 4, 6, 7];
            $page_to_bank = [0 => 16, 1 => 17, 3 => 19, 4 => 20, 6 => 22, 7 => 23];

            // Для подбора страниц берём фактически оставшееся место по счётчику аллокатора
            $available_by_page = [];
            foreach ($allowed_pages as $page) {
                $available_by_page[$page] = isset($this->space[$page]) ? (int)$this->space[$page] : 0;
            }

            // Сортируем треки по убыванию размера (FFD) и выбираем лучшую страницу (Best-Fit)
            usort($items, function($a, $b) { return $b['size'] - $a['size']; });

            $placed = [];
            foreach ($items as $it) {
                $best_page = null;
                $best_leftover = null;
                foreach ($allowed_pages as $page) {
                    if ($available_by_page[$page] >= $it['size']) {
                        $leftover = $available_by_page[$page] - $it['size'];
                        if ($best_leftover === null || $leftover < $best_leftover) {
                            $best_leftover = $leftover;
                            $best_page = $page;
                        }
                    }
                }
                if ($best_page === null) {
                    continue; // Не помещается никуда
                }

                // Запоминаем назначение страницы/банка; адрес вычислим позже проходом по data_flow
                $placed[] = [
                    'order' => $it['order'],
                    'name' => $it['name'],
                    'size' => $it['size'],
                    'page' => $best_page,
                    'bank' => $page_to_bank[$best_page],
                    'addr' => null
                ];
                // Корректируем реальный счётчик свободного места и синхронизируем локальный расчёт
                $allocOk = $this->allocSpace($it['size'], $best_page);
                if ($allocOk === false) {
                    $this->isOverflow = true;
                    // В случае сбоя оставляем available_by_page без изменений, чтобы избежать расхождений
                } else {
                    $available_by_page[$best_page] = isset($this->space[$best_page]) ? (int)$this->space[$best_page] : 0;
                }
            }

            // Сортируем размещённые по адресу внутри страницы и добавляем incbin в data_flow
            $placed_by_page = [];
            foreach ($placed as $p) { $placed_by_page[$p['page']][] = $p; }
            foreach ($placed_by_page as $page => $list) {
                usort($list, function($a, $b) { return $a['addr'] - $b['addr']; });
                $bank_data = [];
                foreach ($list as $pitem) {
                    $label = 'MB_ADDR_' . $pitem['order'];
                    $bank_data[] = sprintf('%s:', $label);
                    $bank_data[] = sprintf('    incbin "res/%s"', $pitem['name']);
                }
                if (!empty($bank_data)) {
                    if (!empty($data_flow[$page])) {
                        $data_flow[$page] = array_merge($data_flow[$page], $bank_data);
                    } else {
                        $data_flow[$page] = $bank_data;
                    }
                }
            }
            // Проставляем точные адреса музыки, проходя по data_flow постфактум
            // Подготовим размеры известных ресурсов
            // Размеры музыки
            foreach ($items as $it) { $resSizes[$it['name']] = $it['size']; }

            $music_addr_map = [];
            foreach ($allowed_pages as $page) {
                if (empty($data_flow[$page])) continue;
                $addr = 0xC000;
                foreach ($data_flow[$page] as $entry) {
                    if (preg_match('/\\borg\\s*\$([0-9A-Fa-f]{4})/', $entry, $m)) {
                        $addr = hexdec($m[1]);
                        continue;
                    }
                    if (preg_match('/\\b(block|ds)\\s+(\\d+)/i', $entry, $m)) {
                        $addr += (int)$m[2];
                        continue;
                    }
                    if (preg_match('/incbin\\s+\"res\/(.+?)\"/i', $entry, $m)) {
                        $fname = $m[1];
                        // Если это музыка, запоминаем адрес старта
                        if (preg_match('/^music\\d+\.(pt2|pt3)$/i', $fname)) {
                            $music_addr_map[$fname] = $addr;
                        }
                        if (isset($resSizes[$fname])) {
                            $addr += (int)$resSizes[$fname];
                        }
                        continue;
                    }
                    if (preg_match('/^\\s*db\\s+[#$]/i', $entry)) { $addr += 1; continue; }
                }
            }

            // Пересобираем MusicBankTable по порядку исходного списка
            usort($placed, function($a, $b) { return $a['order'] - $b['order']; });
            $table_lines = [ 'MusicBankTable:' ];
            foreach ($placed as $p) {
                $label = 'MB_ADDR_' . $p['order'];
                $table_lines[] = sprintf('    dw %s : dw %d', $label, $p['bank']);
            }
            // Обновляем таблицу в параметрах для добавления в шаблон
            $params['musicbank']['asm_table'] = implode("\n", $table_lines);

            // Обновляем mus_kol на реальное число размещённых
            $placed_count = count($placed);
            $mus_kol_value = sprintf('#%02x', 0x30 + $placed_count + 1);
            if (function_exists('preg_replace')) {
                $source_tpl = preg_replace('/(mus_kol\s*\R\s*\tcp )#[0-9A-Fa-f]{2}/', '$1' . $mus_kol_value, $source_tpl);
            }
            // Место под MusicBankTable не нужно выделять, так как таблица размещается в шаблоне
        }

        // Almost done. Finalize.
        if (empty($timeline)) {
            $source_tpl = str_replace('%timeline%', '', $source_tpl);
            $source_tpl = str_replace('%functions%', '', $source_tpl);
            // Включаем обработку прерываний для музыки даже без timeline
            if ($has_music) {
                $source_tpl = str_replace('%if_int_flow%', '', $source_tpl);
                // Учитываем базовый размер обработчика прерываний без записей таймлайна
                $this->allocSpace($this->sizes['Int flow'], 0);
            } else {
                $source_tpl = str_replace('%if_int_flow%', ';', $source_tpl);
            }
        } else {
            $this->allocSpace($this->sizes['Int flow'] + count($timeline) * 8, 0);

            list($timeline, $functions) = $this->generateTimeline($timeline);
            $source_tpl = str_replace('%timeline%', $timeline, $source_tpl);
            $source_tpl = str_replace('%functions%', $functions, $source_tpl);
            $source_tpl = str_replace('%if_int_flow%', '', $source_tpl);
        }

        // Условное добавление laser521.asm и MusicBankTable
        if ($use_laser521) {
            $zip->addFile(PROJECT_ROOT . 'resources/output/sources/laser521.asm', 'sources/laser521.asm');
            $replacement = "\tinclude \"sources/laser521.asm\"";
            // Добавляем MusicBankTable сразу после laser521.asm
            if (isset($params['musicbank']) && isset($params['musicbank']['asm_table']) && isset($params['musicbank']['music_files'])) {
                $replacement .= "\n" . $params['musicbank']['asm_table'];
            }
            $source_tpl = str_replace("%if_laser521%\tinclude \"sources/laser521.asm\"", $replacement, $source_tpl);
        } else {
            $source_tpl = str_replace("%if_laser521%\tinclude \"sources/laser521.asm\"", '', $source_tpl);
        }

        $source_tpl = str_replace('%main_flow%', $this->generateMainFlow($main_flow), $source_tpl);
        $source_tpl = str_replace('%data_flow%', $this->generateDataFlow($data_flow), $source_tpl);
        $zip->addFromString('sources/zapil.asm', $source_tpl);
        

        
        $zip->close();

        return $dest_filename;
    }

    // Новый метод для SlideShow
    function parseSlideShow($zipFilePath) {
        // Доступные банки (16,17,19,20,22,23)
        $banks = [16,17,19,20,22,23];
        $bank_size = 16384; // 16K
        $bank_start = 0xC000;
        // Для банка 0 может быть меньше, если часть занята системой, но для SlideShow используем весь банк

        // Текущее заполнение банков
        $bank_offsets = [];
        foreach ($banks as $b) $bank_offsets[$b] = $bank_start;

        // Открываем архив
        $zip = new ZipArchive();
        if ($zip->open($zipFilePath) !== true) {
            $this->error('Unable to open ZIP archive', __FILE__, __LINE__);
            return false;
        }

        // Собираем список файлов (без директорий, сортируем по размеру для оптимизации)
        $files = [];
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $entry = $zip->getNameIndex($i);
            if (substr($entry, -1) == '/') continue;
            $stat = $zip->statIndex($i);
            $files[] = [
                'name' => $entry,
                'size' => $stat['size'],
                'index' => $i
            ];
        }
        // Сортируем по убыванию размера для оптимального заполнения
        usort($files, function($a, $b) { return $b['size'] - $a['size']; });

        // Распределяем файлы по банкам
        $file_placement = [];
        foreach ($files as $file) {
            $placed = false;
            foreach ($banks as $b) {
                $offset = $bank_offsets[$b];
                $used = $offset - $bank_start;
                if ($used + $file['size'] <= $bank_size) {
                    $file_placement[$file['name']] = [
                        'bank' => $b,
                        'offset' => $offset,
                        'size' => $file['size'],
                        'index' => $file['index']
                    ];
                    $bank_offsets[$b] += $file['size'];
                    $placed = true;
                    break;
                }
            }
            if (!$placed) {
                $this->error('Not enough memory to place file: ' . $file['name'], __FILE__, __LINE__);
                return false;
            }
        }

        // Теперь формируем таблицу по номерам в названиях файлов (1, 2, 3...)
        $table_entries = [];
        $slide_files = [];
        
        // Извлекаем номера из названий файлов и сортируем по ним
        $numbered_files = [];
        foreach ($files as $file) {
            $filename = pathinfo($file['name'], PATHINFO_FILENAME); // убираем расширение
            // Ищем цифры в названии файла
            if (preg_match('/(\d+)/', $filename, $matches)) {
                $number = intval($matches[1]);
                $numbered_files[] = [
                    'name' => $file['name'],
                    'number' => $number,
                    'placement' => $file_placement[$file['name']]
                ];
            }
        }
        
        // Сортируем по номерам (1, 2, 3...)
        usort($numbered_files, function($a, $b) { return $a['number'] - $b['number']; });
        
        // Формируем таблицу в порядке номеров
        foreach ($numbered_files as $file) {
            $placement = $file['placement'];
            $slide_files[] = [
                'name' => $file['name'],
                'bank' => $placement['bank'],
                'offset' => $placement['offset'],
                'size' => $placement['size'],
                'index' => $placement['index'],
                'number' => $file['number']
            ];
        }

        // Сохраняем файлы для дальнейшего включения в проект (в порядке номеров)
        $slide_data = [];
        foreach ($slide_files as $f) {
            $slide_data[] = [
                'name' => $f['name'],
                'bank' => $f['bank'],
                'offset' => $f['offset'],
                'size' => $f['size'],
                'data' => $zip->getFromIndex($f['index']),
                'number' => $f['number']
            ];
        }
        $zip->close();

        return [
            'slide_files' => $slide_data
        ];
    }

    // Новый метод для MusicBank
    function parseMusicBank($zipFilePath) {
        // Доступные банки (16,17,19,20,22,23)
        $banks = [16,17,19,20,22,23];
        $bank_size = 16384; // 16K
        $bank_start = 0xC000;

        // Текущее заполнение банков
        $bank_offsets = [];
        foreach ($banks as $b) $bank_offsets[$b] = $bank_start;

        // Открываем архив
        $zip = new ZipArchive();
        if ($zip->open($zipFilePath) !== true) {
            $this->error('Unable to open ZIP archive', __FILE__, __LINE__);
            return false;
        }

        // Собираем список файлов (только .pt2/.pt3, не более 10 штук)
        $files = [];
        $music_count = 0;
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $entry = $zip->getNameIndex($i);
            if (substr($entry, -1) == '/') continue;
            
            // Проверяем расширение файла
            $extension = strtolower(pathinfo($entry, PATHINFO_EXTENSION));
            if ($extension !== 'pt2' && $extension !== 'pt3') continue;
            
            $music_count++;
            if ($music_count > 10) {
                $this->error('Too many PT2/PT3 files in archive. Maximum 10 files allowed.', __FILE__, __LINE__);
                $zip->close();
                return false;
            }
            
            $stat = $zip->statIndex($i);
            $files[] = [
                'name' => $entry,
                'size' => $stat['size'],
                'index' => $i
            ];
        }
        
        if (empty($files)) {
            $this->error('No PT2/PT3 files found in archive.', __FILE__, __LINE__);
            $zip->close();
            return false;
        }
        
        if (count($files) < 2) {
            $this->error('MusicBank requires at least 2 PT2/PT3 files in archive.', __FILE__, __LINE__);
            $zip->close();
            return false;
        }
        
        // Сортируем по убыванию размера для оптимального заполнения
        usort($files, function($a, $b) { return $b['size'] - $a['size']; });

        // Распределяем файлы по банкам
        $file_placement = [];
        foreach ($files as $file) {
            $placed = false;
            foreach ($banks as $b) {
                $offset = $bank_offsets[$b];
                $used = $offset - $bank_start;
                if ($used + $file['size'] <= $bank_size) {
                    $file_placement[$file['name']] = [
                        'bank' => $b,
                        'offset' => $offset,
                        'size' => $file['size'],
                        'index' => $file['index']
                    ];
                    $bank_offsets[$b] += $file['size'];
                    $placed = true;
                    break;
                }
            }
            if (!$placed) {
                $this->error('Not enough memory to place file: ' . $file['name'], __FILE__, __LINE__);
                return false;
            }
        }

        // Теперь формируем таблицу по номерам в названиях файлов (1, 2, 3...)
        $table_entries = [];
        $music_files = [];
        
        // Извлекаем номера из названий файлов и сортируем по ним
        $numbered_files = [];
        foreach ($files as $file) {
            $filename = pathinfo($file['name'], PATHINFO_FILENAME); // убираем расширение
            // Ищем цифры в названии файла
            if (preg_match('/(\d+)/', $filename, $matches)) {
                $number = intval($matches[1]);
                $numbered_files[] = [
                    'name' => $file['name'],
                    'number' => $number,
                    'placement' => $file_placement[$file['name']]
                ];
            }
        }
        
        // Сортируем по номерам (1, 2, 3...)
        usort($numbered_files, function($a, $b) { return $a['number'] - $b['number']; });
        
        // Формируем таблицу в порядке номеров
        foreach ($numbered_files as $file) {
            $placement = $file['placement'];
            $table_entries[] = sprintf('    dw $%04X : dw %d', $placement['offset'], $placement['bank']);
            $music_files[] = [
                'name' => $file['name'],
                'bank' => $placement['bank'],
                'offset' => $placement['offset'],
                'size' => $placement['size'],
                'index' => $placement['index'],
                'number' => $file['number']
            ];
        }
        $asm_table = "MusicBankTable:\n" . implode("\n", $table_entries);

        // Сохраняем файлы для дальнейшего включения в проект (в порядке номеров)
        $music_data = [];
        foreach ($music_files as $f) {
            $music_data[] = [
                'name' => $f['name'],
                'bank' => $f['bank'],
                'offset' => $f['offset'],
                'size' => $f['size'],
                'data' => $zip->getFromIndex($f['index']),
                'number' => $f['number']
            ];
        }
        $zip->close();

        return [
            'asm_table' => $asm_table,
            'music_files' => $music_data,
            'music_count' => count($music_data)
        ];
    }
}
