<?php

/**
 * TRD File Merger - объединяет файлы внутри TRD образа
 */
class TRDMerger {
    
    /**
     * Создает моноблок из файлов с учетом ограничений TR-DOS (255 секторов максимум)
     * @param string $trdPath Путь к TRD файлу
     * @param array $filesToMerge Массив файлов для объединения [['name' => 'file1.C', 'start' => 0x6000], ...]
     * @param string $outputName Имя выходного объединенного файла
     * @param int $outputStartAddr Начальный адрес для объединенного файла
     * @return bool Успешность операции
     */
    public static function createMonoblock($trdPath, $filesToMerge, $outputName = 'monoblock.C', $outputStartAddr = 0x5C00) {

        
        if (!file_exists($trdPath)) {

            return false;
        }
        
        $trdData = file_get_contents($trdPath);
        if ($trdData === false) {
            return false;
        }
        
        // Каталог TR-DOS занимает сектора 0-7 (2048 байт)
        $directory = substr($trdData, 0, 2048);
        
                 // Собираем информацию о файлах (только из каталога, без копирования данных)
         $filesInfo = [];
         $totalSectors = 0;
        foreach ($filesToMerge as $fileInfo) {
            $fileName = $fileInfo['name'];
            $entryIndex = self::findFileEntry($directory, $fileName);
            if ($entryIndex === false) {

                continue;
            }
            $offset = $entryIndex * 16;
            $entry = substr($directory, $offset, 16);
                                      // Длина в каталоге TR-DOS хранится в СЕКТОРАХ (байт 13)
             $lengthSectors = ord($entry[13]);
             // Старт: байт 15 — номер сектора на дорожке (1..16)
             $startSector = ord($entry[15]);
             
             $filesInfo[] = [
                 'index' => $entryIndex,
                 'offset' => $offset,
                 'length_sectors' => $lengthSectors,
                 'start_sector' => $startSector,
                 'name' => $fileName,
             ];
             $totalSectors += $lengthSectors;
        }
        if (empty($filesInfo)) {
            return false;
        }
        
        // Первый файл должен быть boot.B — расширяем его на всю склейку (не трогаем адреса)
        $bootIndex = self::findFileEntry($directory, 'boot.B');
        if ($bootIndex === false) {

            return false;
        }
        $bootOff = $bootIndex * 16;
        $bootEntry = substr($directory, $bootOff, 16);
                 // Сохраняем хвост записи как есть (байты 9..15), чтобы не трогать адреса/старт
         $bootTail = substr($bootEntry, 9, 7);
        // Старт boot.B: байт 15 — номер сектора на дорожке (1..16)
        $bootStartSector = ord($bootEntry[15]);
        
                 $sectorSize = 256;
         $maxSectors = 255;
         

         
         $firstPartSectors = min($totalSectors, $maxSectors);
         $firstPartBytes = $firstPartSectors * $sectorSize;
         $remainingSectors = max(0, $totalSectors - $firstPartSectors);
         $remainingBytes = $remainingSectors * $sectorSize;
         
         
        
                 // Обновляем ТОЛЬКО байт 13 (длина в секторах), трек и сектор оставляем как было
         $newBootEntry = substr($bootEntry, 0, 13)  // имя/тип + адрес + длина в байтах (0-12) - НЕ ТРОГАЕМ
             . chr($firstPartSectors)                // новая длина в секторах (13)
             . substr($bootEntry, 14, 2);            // трек + сектор (14-15) - НЕ ТРОГАЕМ, оставляем как было
                 // $newBootEntry уже ровно 16 байт (13 + 1 + 2 = 16), точно 16 байт
        
        // Очищаем весь каталог и добавляем только нужные файлы
        $directory = str_repeat(chr(0), 2048);
        
        // Добавляем boot.B в позицию 0
        $directory = substr($directory, 0, 0) . $newBootEntry . substr($directory, 16);
        
        // Принудительно очищаем все остальные позиции
        for ($i = 1; $i < 128; $i++) {
            $off = $i * 16;
            $directory = substr($directory, 0, $off) . str_repeat(chr(0), 16) . substr($directory, $off + 16);
        }
        
        // Принудительно очищаем все позиции после boot.B
        for ($i = 2; $i < 128; $i++) {
            $off = $i * 16;
            $directory = substr($directory, 0, $off) . str_repeat(chr(0), 16) . substr($directory, $off + 16);
        }
        
        // Принудительно очищаем все позиции после второго файла
        for ($i = 3; $i < 128; $i++) {
            $off = $i * 16;
            $directory = substr($directory, 0, $off) . str_repeat(chr(0), 16) . substr($directory, $off + 16);
        }
        
        // Если требуется вторая часть (>255 секторов) — создаем/переиспользуем следующую свободную запись
        if ($remainingSectors > 0) {
            // Ищем свободную запись ПОСЛЕ boot.B, чтобы boot.B оставался первым
            $freeEntryIndex = -1;
            for ($i = $bootIndex + 1; $i < 128; $i++) {
                $off = $i * 16;
                if (ord($directory[$off]) === 0) { $freeEntryIndex = $i; break; }
            }
            // Если не нашли после boot, берем любую свободную
            if ($freeEntryIndex === -1) {
                for ($i = 0; $i < 128; $i++) {
                    $off = $i * 16;
                    if (ord($directory[$off]) === 0) { $freeEntryIndex = $i; break; }
                }
            }
            if ($freeEntryIndex !== -1) {
                                 // Для второй записи: старт = старт boot + количество секторов первой части
                 // boot.B начинается с трека 1, сектор 0, занимает 255 секторов
                 // второй файл начинается с трека 16, сектор 15
                 $secondTrack = 16; // трек 16 (0x10 в hex)
                 $secondStartSector = 15; // сектор 15 (0x0F в hex)
                                 $secondName = str_pad('demo1', 8, ' ', STR_PAD_RIGHT);
                $secondType = 'C';
                                                  // Формируем вторую запись правильно:
                 // байты 0-7: имя (demo1-10)
                 // байт 8: тип (C)
                 // байты 9-10: адрес загрузки (копируем из boot.B)
                 // байты 11-12: длина в байтах (remainingSectors * 256)
                 // байт 13: длина в секторах (remainingSectors)
                 // байт 14: трек (копируем из boot.B)
                 // байт 15: сектор на дорожке (secondStartSector)
                                   $secondEntry = $secondName
                       . $secondType
                       . substr($bootEntry, 9, 2)  // адрес загрузки из boot.B (байты 9-10)
                       . chr($remainingSectors * 256 & 0xFF)  // длина в байтах (младший, байт 11)
                       . chr(($remainingSectors * 256 >> 8) & 0xFF)  // длина в байтах (старший, байт 12)
                       . chr($remainingSectors)  // длина в секторах (байт 13)
                                               . chr($secondStartSector)  // сектор на дорожке (байт 14)
                        . chr($secondTrack);  // новый трек (байт 15)
                                 // $secondEntry уже ровно 16 байт (8 + 1 + 2 + 2 + 1 + 2 + 1 = 17), но substr обрежет до 16
                                 // вставляем второй файл в позицию 1
                 $directory = substr($directory, 0, 16) . $secondEntry . substr($directory, 32);
            }
        }
        
        // Подсчитываем реальное количество файлов в итоговом каталоге
        $fileCount = 0;
        
        for ($i = 0; $i < 128; $i++) {
            $off = $i * 16;
            $entry = substr($directory, $off, 16);
            if (ord($entry[0]) !== 0) { // если первый байт не 0, значит файл есть
                $name = rtrim(substr($entry, 0, 8), " \0");
                $type = $entry[8];
                
                $fileCount++;
            }
        }

        // Записываем обновленный каталог (2048 байт) обратно в образ и корректируем служебный сектор 8
        
        
        // Берем данные строго с байта 2048 (сектор 8) до конца
        $dataSection = substr($trdData, 2048);
        $newTrdData = $directory . $dataSection;
        
        // Служебный сектор 8/9: смещение #8E4 (2276) содержит количество файлов
        if (strlen($newTrdData) >= 2277) {
            $newTrdData[2276] = chr($fileCount);
        }
        
        // Проверяем размер перед записью
        $expectedSize = 655360; // 2560 секторов * 256 байт
        if (strlen($newTrdData) !== $expectedSize) {
            // Размер не соответствует ожидаемому
        }
        
        $result = file_put_contents($trdPath, $newTrdData) !== false;
        
        return $result;
    }
    
    /**
     * Находит запись файла в каталоге TRD
     */
    private static function findFileEntry($directory, $fileName) {
        for ($i = 0; $i < 128; $i++) {
            $offset = $i * 16;
            $entry = substr($directory, $offset, 16);
            if ($entry === '' || strlen($entry) < 16) { continue; }
            if (ord($entry[0]) === 0) { continue; }
            $name = rtrim(substr($entry, 0, 8), " \0");
            $type = $entry[8];
            if ($name . '.' . $type === $fileName) {
                return $i; // возвращаем индекс записи
            }
        }
        return false;
    }
    
    /**
     * Извлекает данные файла из TRD
     */
    private static function extractFileData($trdData, $fileEntry) {
        $length = ord($fileEntry[9]) + (ord($fileEntry[10]) << 8);
        $startSector = ord($fileEntry[13]) + (ord($fileEntry[14]) << 8);
        
        if ($startSector === 0 || $length === 0) {
            return false;
        }
        
        $dataOffset = $startSector * 256;
        return substr($trdData, $dataOffset, $length);
    }
    
    /**
     * Добавляет файл в TRD образ
     */
    private static function addFileToTRD($trdPath, $fileName, $fileData, $startAddr) {
        $trdData = file_get_contents($trdPath);
        $directory = substr($trdData, 0, 256);
        
        // Находим свободную запись в каталоге
        $freeEntryIndex = -1;
        for ($i = 0; $i < 128; $i++) {
            $offset = $i * 16;
            $entry = substr($directory, $offset, 16);
            
            if (ord($entry[0]) === 0) {
                $freeEntryIndex = $i;
                break;
            }
        }
        
        if ($freeEntryIndex === -1) {
            return false; // Нет свободного места в каталоге
        }
        
        // Находим свободное место для данных
        $freeSector = self::findFreeSector($directory);
        if ($freeSector === -1) {
            return false; // Нет свободного места для данных
        }
        
        // Создаем новую запись в каталоге
        $name = substr($fileName, 0, strpos($fileName, '.'));
        $type = substr($fileName, strpos($fileName, '.') + 1);
        
        $newEntry = str_pad($name, 8, ' ', STR_PAD_RIGHT);
        $newEntry .= $type;
        $newEntry .= chr(strlen($fileData) & 0xFF);
        $newEntry .= chr((strlen($fileData) >> 8) & 0xFF);
        $newEntry .= chr($startAddr & 0xFF);
        $newEntry .= chr(($startAddr >> 8) & 0xFF);
        $newEntry .= chr($startAddr & 0xFF); // Start address = load address
        $newEntry .= chr(($startAddr >> 8) & 0xFF);
        $newEntry .= chr($freeSector & 0xFF);
        $newEntry .= chr(($freeSector >> 8) & 0xFF);
        $newEntry .= str_repeat(chr(0), 6); // Reserved
        
        // Вставляем запись в каталог
        $newDirectory = substr($directory, 0, $freeEntryIndex * 16) . 
                       $newEntry . 
                       substr($directory, ($freeEntryIndex + 1) * 16);
        
        // Вставляем данные файла
        $dataOffset = $freeSector * 256;
        $newTrdData = substr($trdData, 0, $dataOffset) . 
                     $fileData . 
                     substr($trdData, $dataOffset + strlen($fileData));
        
        // Обновляем каталог
        $newTrdData = $newDirectory . substr($newTrdData, 256);
        
        // Сохраняем обновленный TRD
        return file_put_contents($trdPath, $newTrdData) !== false;
    }
    
    /**
     * Находит свободный сектор для данных
     */
    private static function findFreeSector($directory) {
        $usedSectors = [];
        
        // Собираем все используемые секторы
        for ($i = 0; $i < 128; $i++) {
            $offset = $i * 16;
            $entry = substr($directory, $offset, 16);
            
            if (ord($entry[0]) === 0) {
                continue;
            }
            
            $length = ord($entry[9]) + (ord($entry[10]) << 8);
            $startSector = ord($entry[13]) + (ord($entry[14]) << 8);
            
            if ($startSector > 0 && $length > 0) {
                $sectorsNeeded = ceil($length / 256);
                for ($s = 0; $s < $sectorsNeeded; $s++) {
                    $usedSectors[$startSector + $s] = true;
                }
            }
        }
        
        // Ищем первый свободный сектор
        for ($sector = 1; $sector < 1600; $sector++) {
            if (!isset($usedSectors[$sector])) {
                return $sector;
            }
        }
        
        return -1; // Нет свободного места
    }
    
    /**
     * Разбивает моноблок на несколько файлов с учетом ограничения 255 секторов
     */
    private static function splitMonoblock($trdPath, $mergedData, $originalFiles, $startAddr) {
        $sectorSize = 256;
        $maxSectors = 255;
        $maxSize = $maxSectors * $sectorSize;
        
        // Разбиваем данные
        $firstPart = substr($mergedData, 0, $maxSize);
        $secondPart = substr($mergedData, $maxSize);
        
        // Находим boot.B в исходных файлах
        $bootFile = null;
        foreach ($originalFiles as $file) {
            if (strpos($file['name'], 'boot.B') !== false) {
                $bootFile = $file;
                break;
            }
        }
        
        if ($bootFile) {
            // Обновляем boot.B с первой частью данных (сохраняем атрибуты)
            $result1 = self::updateBootFile($trdPath, $firstPart);
        } else {
            // Создаем новый файл для первой части
            $result1 = self::addFileToTRD($trdPath, 'boot.B', $firstPart, $startAddr);
        }
        
        // Добавляем вторую часть как отдельный файл
        $result2 = self::addFileToTRD($trdPath, 'demo1-10.C', $secondPart, 0x6000);
        
        // Обновляем количество файлов в секторе 9
        self::updateFileCount($trdPath);
        
        return $result1 && $result2;
    }
    
    /**
     * Обновляет boot.B файл новыми данными, сохраняя атрибуты
     */
    private static function updateBootFile($trdPath, $newData) {
        $trdData = file_get_contents($trdPath);
        $directory = substr($trdData, 0, 256);
        
        // Находим boot.B в каталоге
        for ($i = 0; $i < 128; $i++) {
            $offset = $i * 16;
            $entry = substr($directory, $offset, 16);
            
            if (ord($entry[0]) === 0) continue;
            
            $name = trim(substr($entry, 0, 8));
            $type = $entry[8];
            
            if ($name . '.' . $type === 'boot.B') {
                
                // Сохраняем атрибуты boot.B
                $loadAddr = ord($entry[11]) + (ord($entry[12]) << 8);
                $execAddr = ord($entry[13]) + (ord($entry[14]) << 8);
                $startSector = ord($entry[15]) + (ord($entry[16]) << 8);
                
                // Обновляем длину и количество секторов
                $newLength = strlen($newData);
                $newSectors = ceil($newLength / 256);
                
                // Создаем новую запись
                $newEntry = substr($entry, 0, 9); // Имя и тип
                $newEntry .= chr($newLength & 0xFF);
                $newEntry .= chr(($newLength >> 8) & 0xFF);
                $newEntry .= chr($loadAddr & 0xFF);
                $newEntry .= chr(($loadAddr >> 8) & 0xFF);
                $newEntry .= chr($execAddr & 0xFF);
                $newEntry .= chr(($execAddr >> 8) & 0xFF);
                $newEntry .= chr($startSector & 0xFF);
                $newEntry .= chr(($startSector >> 8) & 0xFF);
                $newEntry .= str_repeat(chr(0), 6); // Резерв
                
                // Обновляем каталог
                $newDirectory = substr($directory, 0, $offset) . 
                               $newEntry . 
                               substr($directory, $offset + 16);
                
                // Обновляем данные
                $dataOffset = $startSector * 256;
                $newTrdData = substr($trdData, 0, $dataOffset) . 
                             $newData . 
                             substr($trdData, $dataOffset + strlen($newData));
                
                // Обновляем каталог
                $newTrdData = $newDirectory . substr($newTrdData, 256);
                
                return file_put_contents($trdPath, $newTrdData) !== false;
            }
        }
        
        return false;
    }
    
    /**
     * Обновляет количество файлов в секторе 9
     */
    private static function updateFileCount($trdPath) {
        $trdData = file_get_contents($trdPath);
        $directory = substr($trdData, 0, 256);
        
        // Подсчитываем количество файлов
        $fileCount = 0;
        for ($i = 0; $i < 128; $i++) {
            $offset = $i * 16;
            $entry = substr($directory, $offset, 16);
            if (ord($entry[0]) !== 0) {
                $fileCount++;
            }
        }
        
        // Обновляем сектор 8/9: смещение #E4 (228) содержит количество файлов
        if (strlen($trdData) >= 228) {
            $trdData[228] = chr($fileCount);
        }
        
        return file_put_contents($trdPath, $trdData) !== false;
    }
    
    /**
     * Удаляет файлы из TRD образа
     */
    public static function removeFiles($trdPath, $fileNames) {
        if (!file_exists($trdPath)) {
            return false;
        }
        
        $trdData = file_get_contents($trdPath);
        $directory = substr($trdData, 0, 256);
        $modified = false;
        
        foreach ($fileNames as $fileName) {
            for ($i = 0; $i < 128; $i++) {
                $offset = $i * 16;
                $entry = substr($directory, $offset, 16);
                
                if (ord($entry[0]) === 0) {
                    continue;
                }
                
                $name = trim(substr($entry, 0, 8));
                $type = $entry[8];
                
                if ($name . '.' . $type === $fileName) {
                    // Очищаем запись
                    $directory = substr($directory, 0, $offset) . 
                                str_repeat(chr(0), 16) . 
                                substr($directory, $offset + 16);
                    $modified = true;
                    break;
                }
            }
        }
        
        if ($modified) {
            $newTrdData = $directory . substr($trdData, 256);
            $result = file_put_contents($trdPath, $newTrdData) !== false;
            
            // Обновляем количество файлов в секторе 9
            if ($result) {
                self::updateFileCount($trdPath);
            }
            
            return $result;
        }
        
        return true;
    }
}
