<?php

/**
 * @desc Extracting sources from ZIP-archive, compiling, and adding into archive
 * @param $outputDir string Output directory
 * @param $projectName string Project filename
 * @param $zipPath string Path to ZIP-archive with project files
 * @return false|string
 */
function CompileCode($outputDir, $projectName, $zipPath, $trdMergeConfig = null) {

    $projectDir = $outputDir . '/' . $projectName;
    $sjasmplus_log = '';

    if (!file_exists($projectDir)) {
        mkdir($projectDir, 0777, true);
    } else {
        recurseRmdir($projectDir);
    }

    $zip = new ZipArchive;
    if ($zip->open($zipPath) !== true) {
        return false;
    }
    if (!$zip->extractTo($projectDir)) {
        return false;
    }
    
    // --- Copy additional files to project directory BEFORE compilation ---
    $dehrustPath = PROJECT_ROOT . 'resources/output/sources/dehrust.bin';
    if (file_exists($dehrustPath)) {
        copy($dehrustPath, $projectDir . '/dehrust.bin');
    }

    // --- Capture sjasmplus output ---
    $cmd = escapeshellcmd(PROJECT_ROOT . 'resources/output/sjasmplus.exe') .
        ' --inc=' . escapeshellarg($projectDir . '/.') .
        ' --inc=' . escapeshellarg($projectDir . '/sources/.') .
        ' ' . escapeshellarg($projectDir . '/sources/zapil.asm');
    $descriptorspec = [
        1 => ['pipe', 'w'], // stdout
        2 => ['pipe', 'w'], // stderr
    ];
    $process = proc_open($cmd, $descriptorspec, $pipes, $projectDir);

    if (is_resource($process)) {
        $first_compile_log = stream_get_contents($pipes[1]);
        $first_compile_log .= stream_get_contents($pipes[2]);
        fclose($pipes[1]);
        fclose($pipes[2]);
        proc_close($process);
        
        // Always show first compile log to get page size information
        $sjasmplus_log .= "First compilation:\n" . $first_compile_log;
    }


    
        // --- Pack .C files with hrust13 AFTER first compilation ---
    $hrust13Path = PROJECT_ROOT . 'resources/output/hrust13.exe';
    if (file_exists($hrust13Path)) {
        // Pack all .C files with hrust13 (except pages with slideshow)
        for ($page = 0; $page <= 7; $page++) {
            $sourceFile = $projectDir . "/page{$page}.c";
            $packedFile = $projectDir . "/page{$page}p.c";
                                        $slideshowMarker = PROJECT_ROOT . "var/cache/zapilyator_page{$page}_slideshow";
                            // Пропускаем страницы со слайдами - они уже упакованы через laser521.exe
                            if (file_exists($slideshowMarker)) {
                                $sjasmplus_log .= "\nSkipping page{$page}.c (contains slideshow images)";
                                // Не удаляем маркер здесь, он нужен для второго цикла
                                continue;
                            }
            
            if (file_exists($sourceFile)) {
                $cmd = escapeshellcmd($hrust13Path) . 
                       ' ' . escapeshellarg($sourceFile) . 
                       ' ' . escapeshellarg($packedFile);
                $output = array();
                $returnCode = 0;
                exec($cmd, $output, $returnCode);
                
                if ($returnCode === 0 && file_exists($packedFile)) {
                    $originalSize = filesize($sourceFile);
                    $packedSize = filesize($packedFile);
                    $compressionRatio = round((1 - $packedSize / $originalSize) * 100, 1);
                    $sjasmplus_log .= "\nPacked page{$page}.c: {$originalSize} -> {$packedSize} bytes ({$compressionRatio}% compression)";
                } else {
                    $sjasmplus_log .= "\nFailed to pack page{$page}.c (return code: $returnCode)";
                }
            }
        }
        
        // Modify builder.asm to use packed files and recompile
        $builderPath = $projectDir . '/sources/builder.asm';
        if (file_exists($builderPath)) {
            $builderContent = file_get_contents($builderPath);
            
            // Replace savetrd commands to use fixed addresses and packed files (only for existing packed files)
            $replacedCount = 0;
            for ($page = 0; $page <= 7; $page++) {
                $packedFile = $projectDir . "/page{$page}p.c";
                $slideshowMarker = PROJECT_ROOT . "var/cache/zapilyator_page{$page}_slideshow";
                
                if (file_exists($packedFile)) {
                    $packedSize = filesize($packedFile);
                    $startAddr = $page == 0 ? "#6000" : "#C000";
                    
                    // Для страниц со слайдами используем оригинальные адреса (они уже в data_flow)
                    if (file_exists($slideshowMarker)) {
                        $oldPattern = "savetrd TRD_FILENAME, \"{$page}.C\",page{$page}s, page{$page}e-page{$page}s";
                        $newPattern = "savetrd TRD_FILENAME, \"{$page}.C\",page{$page}s, page{$page}e-page{$page}s ; slideshow page";
                        
                        $newContent = str_replace($oldPattern, $newPattern, $builderContent);
                        if ($newContent !== $builderContent) {
                            $replacedCount++;
                            $builderContent = $newContent;
                        }
                        
                        // Не удаляем маркер здесь, он нужен для второго цикла
                    } else {
                        $oldPattern = "savetrd TRD_FILENAME, \"{$page}.C\",page{$page}s, page{$page}e-page{$page}s";
                        $newPattern = "savetrd TRD_FILENAME, \"{$page}.C\",page{$page}p_start, page{$page}p_end-page{$page}p_start";
                        
                        $newContent = str_replace($oldPattern, $newPattern, $builderContent);
                        if ($newContent !== $builderContent) {
                            $replacedCount++;
                            $builderContent = $newContent;
                        }
                    }
                }
            }
            $sjasmplus_log .= "\nReplaced $replacedCount savetrd commands total";
            
            // Replace page loading blocks to use packed file addresses and sizes
            for ($page = 0; $page <= 7; $page++) {
                $packedFile = $projectDir . "/page{$page}p.c";
                $slideshowMarker = PROJECT_ROOT . "var/cache/zapilyator_page{$page}_slideshow";
                
                // Если это страница со слайдами, убираем команду call depack (пропускаем распаковку)
                if (file_exists($slideshowMarker)) {
                    // Ищем call depack только в блоке соответствующей страницы
                    $pageBlockPattern = "ifdef _page{$page}";
                    
                    // Ищем блок страницы
                    $pageBlockStart = strpos($builderContent, $pageBlockPattern);
                    if ($pageBlockStart !== false) {
                        // Ищем конец блока (следующий ifdef _page или endif)
                        $pageBlockEnd = $pageBlockStart;
                        $searchStart = $pageBlockStart + strlen($pageBlockPattern);
                        
                        // Ищем следующий ifdef _page или endc
                        while (true) {
                            $nextPagePos = strpos($builderContent, 'ifdef _page', $searchStart);
                            $endcPos = strpos($builderContent, 'endc', $searchStart);
                            
                            if ($nextPagePos !== false && $endcPos !== false) {
                                // Берем ближайший
                                $pageBlockEnd = min($nextPagePos, $endcPos);
                                break;
                            } elseif ($nextPagePos !== false) {
                                $pageBlockEnd = $nextPagePos;
                                break;
                            } elseif ($endcPos !== false) {
                                $pageBlockEnd = $endcPos;
                                break;
                            } else {
                                // Достигли конца файла
                                $pageBlockEnd = strlen($builderContent);
                                break;
                            }
                            $searchStart = $pageBlockEnd + 1;
                        }
                        
                        // Извлекаем блок страницы
                        $pageBlock = substr($builderContent, $pageBlockStart, $pageBlockEnd - $pageBlockStart);
                        
                        // Проверяем, есть ли call depack в блоке
                        $callDepackCount = substr_count($pageBlock, 'call depack');
                        if ($callDepackCount > 0) {
                            // Заменяем call depack на nop только в этом блоке
                            $newPageBlock = str_replace('call depack', 'nop', $pageBlock);
                            
                            // Заменяем блок в основном содержимом
                            $builderContent = str_replace($pageBlock, $newPageBlock, $builderContent);
                        }
                    }
                } elseif (file_exists($packedFile)) {
                    $packedSize = filesize($packedFile);
                    
                    // Replace ld hl, pageXs with packed file address
                    $oldPattern = "ld hl, page{$page}s";
                    $newPattern = "ld hl, page{$page}p_start";
                    $builderContent = str_replace($oldPattern, $newPattern, $builderContent);
                    
                    // Replace size calculations with packed file size
                    $oldPattern = "if low (page{$page}e - page{$page}s) == 0";
                    $newPattern = "if low (page{$page}p_end - page{$page}p_start) == 0";
                    $builderContent = str_replace($oldPattern, $newPattern, $builderContent);
                    
                    $oldPattern = "ld bc, (page{$page}e - page{$page}s)/256*256 + 5";
                    $newPattern = "ld bc, (page{$page}p_end - page{$page}p_start)/256*256 + 5";
                    $builderContent = str_replace($oldPattern, $newPattern, $builderContent);
                    
                    $oldPattern = "ld bc, (page{$page}e - page{$page}s + 256)/256*256 + 5";
                    $newPattern = "ld bc, (page{$page}p_end - page{$page}p_start + 256)/256*256 + 5";
                    $builderContent = str_replace($oldPattern, $newPattern, $builderContent);
                    
                    // Для отладки: показываем содержимое блока page0
                    if ($page == 0) {
                        $pageBlockPattern = "ifdef _page0";
                        $pageBlockStart = strpos($builderContent, $pageBlockPattern);
                        if ($pageBlockStart !== false) {
                            $searchStart = $pageBlockStart + strlen($pageBlockPattern);
                            $nextPagePos = strpos($builderContent, 'ifdef _page', $searchStart);
                            $endifPos = strpos($builderContent, 'endif', $searchStart);
                            
                            if ($nextPagePos !== false && $endifPos !== false) {
                                $pageBlockEnd = min($nextPagePos, $endifPos);
                            } elseif ($nextPagePos !== false) {
                                $pageBlockEnd = $nextPagePos;
                            } elseif ($endifPos !== false) {
                                $pageBlockEnd = $endifPos;
                            } else {
                                $pageBlockEnd = strlen($builderContent);
                            }
                            
                            $pageBlock = substr($builderContent, $pageBlockStart, $pageBlockEnd - $pageBlockStart);
                        }
                    }
                }
            }
            
            // Add packed data sections at the end of the file (before ENDMODULE)
            $packedDataSections = "\n; --- Packed data sections ---\n";
            $includedCount = 0;
            for ($page = 0; $page <= 7; $page++) {
                $packedFile = $projectDir . "/page{$page}p.c";
                $slideshowMarker = PROJECT_ROOT . "var/cache/zapilyator_page{$page}_slideshow";
                
                if (file_exists($packedFile)) {
                    // Пропускаем страницы со слайдами - они уже добавлены в zapilyator.php
                    if (file_exists($slideshowMarker)) {
                        $packedDataSections .= "\t; page {$page} - slideshow data already included\n";
                        // Не удаляем маркер здесь, он нужен для второго цикла
                    } else {
                        $packedDataSections .= "\tpage {$page}\n";
                        $packedDataSections .= "\torg #" . ($page == 0 ? "6000" : "C000") . "\n";
                        $packedDataSections .= "page{$page}p_start:\n\tincbin \"page{$page}p.c\"\npage{$page}p_end:\n\n";
                        $includedCount++;
                    }
                }
            }
            $sjasmplus_log .= "\nIncluded $includedCount packed files in TRD";
            
            // Insert packed data sections at the very beginning of the file
            $builderContent = $packedDataSections . $builderContent;
            

            

            
            // Fix depack calls using absolute address #5d40 + offset
            $builderContent = str_replace('call depack', 'call #5d40 + (depack-start)', $builderContent);
            
            file_put_contents($builderPath, $builderContent);
            

            
            // Recompile with packed files
            $cmd = escapeshellcmd(PROJECT_ROOT . 'resources/output/sjasmplus.exe') .
                ' --inc=' . escapeshellarg($projectDir . '/.') .
                ' --inc=' . escapeshellarg($projectDir . '/sources/.') .
                ' ' . escapeshellarg($projectDir . '/sources/zapil.asm');
            $descriptorspec = [
                1 => ['pipe', 'w'], // stdout
                2 => ['pipe', 'w'], // stderr
            ];
            $process = proc_open($cmd, $descriptorspec, $pipes, $projectDir);
            if (is_resource($process)) {
                $recompile_log = stream_get_contents($pipes[1]);
                $recompile_log .= stream_get_contents($pipes[2]);
                fclose($pipes[1]);
                fclose($pipes[2]);
                proc_close($process);
                
                // Always add recompile log to get page size information
                $sjasmplus_log .= "\n" . $recompile_log;
            }
            
            // Add final statistics
            $totalOriginalSize = 0;
            $totalPackedSize = 0;
            for ($page = 0; $page <= 7; $page++) {
                $sourceFile = $projectDir . "/page{$page}.c";
                $packedFile = $projectDir . "/page{$page}p.c";
                if (file_exists($sourceFile)) {
                    $totalOriginalSize += filesize($sourceFile);
                }
                if (file_exists($packedFile)) {
                    $totalPackedSize += filesize($packedFile);
                }
            }
            if ($totalOriginalSize > 0) {
                $totalCompressionRatio = round((1 - $totalPackedSize / $totalOriginalSize) * 100, 1);
                $sjasmplus_log .= "\nTotal compression: {$totalOriginalSize} -> {$totalPackedSize} bytes ({$totalCompressionRatio}% saved)";
            }
        }
    } else {
        
    }
    
    $compileResult = (file_exists($projectDir . '/zapil.sna') || file_exists($projectDir . '/zapil.trd'));
    
    // TRD file merging (only if trdMergeConfig is provided AND user enabled it)
    
    if (file_exists($projectDir . '/zapil.trd') && $trdMergeConfig !== null && isset($_POST['trd_merge_enabled']) && $_POST['trd_merge_enabled'] == '1') {

        
        require_once PROJECT_ROOT . 'include/helpers/trd_merger.php';
        $sjasmplus_log .= "\nTRD merge config found: " . json_encode($trdMergeConfig);
        
        foreach ($trdMergeConfig as $mergeGroup) {
            $filesToMerge = [];
            
            // Поддерживаем разные типы конфигурации
            if (isset($mergeGroup['pages'])) {
                // Склейка по номерам страниц
                foreach ($mergeGroup['pages'] as $page) {
                    $packedFile = $projectDir . "/page{$page}p.c";
                    if (file_exists($packedFile)) {
                        $filesToMerge[] = [
                            'name' => "{$page}.C",
                            'start' => $page == 0 ? 0x6000 : 0xC000
                        ];
                    }
                }
            } elseif (isset($mergeGroup['files'])) {
                // Прямое указание файлов
                $filesToMerge = $mergeGroup['files'];
            }
            
            if (!empty($filesToMerge)) {
                $outputName = $mergeGroup['output'] ?? 'merged.C';
                $outputAddr = $mergeGroup['address'] ?? 0x6000;
                $removeOriginal = $mergeGroup['remove_original'] ?? false;
                $sjasmplus_log .= "\nAttempting to merge " . count($filesToMerge) . " files...";
                
                $mergeResult = TRDMerger::createMonoblock(
                    $projectDir . '/zapil.trd',
                    $filesToMerge,
                    $outputName,
                    $outputAddr
                );
                
                if ($mergeResult) {
                    $sjasmplus_log .= "\nTRD files merged: " . count($filesToMerge) . " files -> {$outputName}";
                    
                    if ($removeOriginal) {
                        // Удаляем все файлы кроме boot.B (он будет обновлен)
                        $filesToRemove = [];
                        foreach ($filesToMerge as $file) {
                            if ($file['name'] !== 'boot.B') {
                                $filesToRemove[] = $file['name'];
                            }
                        }
                        if (!empty($filesToRemove)) {
                            $sjasmplus_log .= "\nRemoving " . count($filesToRemove) . " original files: " . implode(', ', $filesToRemove);
                            $removeResult = TRDMerger::removeFiles($projectDir . '/zapil.trd', $filesToRemove);
                            if ($removeResult) {
                                $sjasmplus_log .= " (removed " . count($filesToRemove) . " original files)";
                            } else {
                                $sjasmplus_log .= " (failed to remove original files)";
                            }
                        }
                    }
                } else {
                    $sjasmplus_log .= "\nWarning: TRD file merging failed for {$outputName}";
                }
            }
        }
    }
    
    if (file_exists($projectDir . '/zapil.sna')) {
        $zip->addFile($projectDir . '/zapil.sna', 'zapil.sna');
    }
    if (file_exists($projectDir . '/zapil.trd')) {
        $zip->addFile($projectDir . '/zapil.trd', 'zapil.trd');
    }
    
    // Добавляем лог компиляции в архив
    $zip->addFromString('compile_log.txt', $sjasmplus_log);
    
    $zip->close();


    return array('result' => $compileResult, 'log' => $sjasmplus_log);
}

function recurseRmdir($dir) {
    $files = array_diff(scandir($dir), array('.', '..'));
    foreach ($files as $file) {
        (is_dir("$dir/$file") && !is_link("$dir/$file")) ? recurseRmdir("$dir/$file") : unlink("$dir/$file");
    }
    return rmdir($dir);
}

